/************************************************************************************************************\

Module Name:    BufferQueue.c

Description:    Queue for Liberatus buffers.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include <time.h>
#include <errno.h>
#include "BufferQueue.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       BufQ_Init

Description:    Initialize the buffer queue. This must be the first function called with BufferQueue
                structure.

\************************************************************************************************************/
LStatus BufQ_Init(BufferQueue* poBufferQ)
{
    BufQ_Cleanup(poBufferQ);

    LStatus eStatus = (poBufferQ != MNULL) ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LMutex_Create(&(poBufferQ->hMutex));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LCondition_Create(&(poBufferQ->hNotEmptyCond));
    }
    else
    {
        BufQ_Cleanup(poBufferQ);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       BufQ_Cleanup

Description:    Free all resources used by the buffer queue. This must be the last function called with
                BufferQueue structure.

\************************************************************************************************************/
void BufQ_Cleanup(BufferQueue* poBufferQ)
{
    if (poBufferQ != MNULL)
    {
        if (poBufferQ->hNotEmptyCond != MNULL)
        {
            LCondition_Destroy(poBufferQ->hNotEmptyCond);
        }

        if (poBufferQ->hMutex != MNULL)
        {
            LMutex_Destroy(poBufferQ->hMutex);
        }

        if (poBufferQ->aoElement != MNULL)
        {
            free(poBufferQ->aoElement);
        }

        poBufferQ->hMutex           = MNULL;
        poBufferQ->hNotEmptyCond    = MNULL;
        poBufferQ->aoElement        = MNULL;
        poBufferQ->uiElementCount   = 0;
        poBufferQ->uiPushIdx        = 0;
        poBufferQ->uiPopIdx         = 0;
    }
}

/************************************************************************************************************\

Function:       BufQ_IsEmpty

\************************************************************************************************************/
static MBOOL BufQ_IsEmpty(const BufferQueue* poBufferQ)
{
    return (poBufferQ->uiPopIdx == poBufferQ->uiPushIdx) ? MTRUE : MFALSE;
}

/************************************************************************************************************\

Function:       BufQ_IsFull

\************************************************************************************************************/
static MBOOL BufQ_IsFull(const BufferQueue* poBufferQ)
{
    return ((poBufferQ->uiPushIdx - poBufferQ->uiPopIdx) >= poBufferQ->uiElementCount) ? MTRUE : MFALSE;
}

/************************************************************************************************************\

Function:       BufQ_GetSize

Description:    Get the number of elements in the queue.

\************************************************************************************************************/
MUINT BufQ_GetSize(BufferQueue* poBufferQ)
{
    LMutex_Lock(poBufferQ->hMutex);

    MUINT uiSize = poBufferQ->uiPushIdx - poBufferQ->uiPopIdx;

    LMutex_Unlock(poBufferQ->hMutex);

    return uiSize;
}

/************************************************************************************************************\

Function:       BufQ_IncreaseSize

Description:    Private function to increase the number of elements into the buffer queue.

\************************************************************************************************************/
static MBOOL BufQ_IncreaseSize(BufferQueue* poBufferQ)
{
    MUINT uiNewElementCount = max(2 * poBufferQ->uiElementCount, 32);

    // Need more than 8K buffers! There is likely a problem...
    if (uiNewElementCount > (8 * 1024))
    {
        MsgLogErr("ERROR! New buffer queue size is too big (%u bytes). ", uiNewElementCount);
        return MFALSE;
    }

    BufferQueueElement* aoNewElement = malloc(uiNewElementCount * sizeof(BufferQueueElement));

    if (aoNewElement != MNULL)
    {
        MUINT uiFirstBlockSize  = min(poBufferQ->uiElementCount, poBufferQ->uiPushIdx)
                                  - poBufferQ->uiPopIdx;
        MUINT uiSecondBlockSize = poBufferQ->uiPushIdx - poBufferQ->uiPopIdx - uiFirstBlockSize;

        if (uiFirstBlockSize != 0)
        {
            memcpy(aoNewElement,
                   poBufferQ->aoElement + poBufferQ->uiPopIdx,
                   uiFirstBlockSize * sizeof(BufferQueueElement));
        }

        if (uiSecondBlockSize != 0)
        {
            memcpy(aoNewElement + uiFirstBlockSize,
                   poBufferQ->aoElement,
                   uiSecondBlockSize * sizeof(BufferQueueElement));
        }

        if (poBufferQ->aoElement != MNULL)
        {
            free(poBufferQ->aoElement);
        }

        poBufferQ->aoElement        = aoNewElement;
        poBufferQ->uiPopIdx         = 0;
        poBufferQ->uiPushIdx        = uiFirstBlockSize + uiSecondBlockSize;
        poBufferQ->uiElementCount   = uiNewElementCount;

        return MTRUE;
    }
    else
    {
        MsgLogErr("ERROR! Cannot allocate data to increase the queue to %u elements.", uiNewElementCount);
        return MFALSE;
    }
}

/************************************************************************************************************\

Function:       BufQ_Push

Description:    Put the given buffer to the queue tail. If the queue is full, the size will be increased
                automatically.

Parameters:     poBufferQ       The buffer queue
                poBufferInfo    Buffer informations
                uiTag           Submission tag
                hDevThread      Liberatus device thread that has generated the submission tag.

Return:         MTRUE on succeed.
                MFALSE if the queue is full and cannot be increased.

\************************************************************************************************************/
MBOOL BufQ_Push(
        BufferQueue*            poBufferQ,
        BufferInfo*             poBufferInfo,
        MUINT64                 uiTag,
        LDeviceThread_Handle    hDevThread)
{
    MBOOL bDone = MFALSE;

    MsgLog(6, "Buffer[%u], Tag= %ld, DT= %p", poBufferInfo->uiId, uiTag, hDevThread);

    LMutex_Lock(poBufferQ->hMutex);

    MBOOL bWasEmpty = BufQ_IsEmpty(poBufferQ);

    if (BufQ_IsFull(poBufferQ))
    {
        BufQ_IncreaseSize(poBufferQ);
    }

    if (!BufQ_IsFull(poBufferQ))
    {
        MUINT uiPushIdx = poBufferQ->uiPushIdx;

        if (uiPushIdx >= poBufferQ->uiElementCount)
        {
            uiPushIdx -= poBufferQ->uiElementCount;
        }

        poBufferQ->aoElement[uiPushIdx].poBufferInfo     = poBufferInfo;
        poBufferQ->aoElement[uiPushIdx].uiTag            = uiTag;
        poBufferQ->aoElement[uiPushIdx].hDeviceThread    = hDevThread;

        poBufferQ->uiPushIdx++;

        if (!poBufferInfo->bEndOfStream)
        {
            poBufferQ->uiLastTimestampUsec = poBufferInfo->uiTimestampUsec;
        }

        bDone = MTRUE;
    }

    LMutex_Unlock(poBufferQ->hMutex);

    if (bWasEmpty)
    {
        LCondition_Signal(poBufferQ->hNotEmptyCond);
    }

    return bDone;
}

/************************************************************************************************************\

Function:       BufQ_Pop

Description:    Get a buffer from the queue head.

Parameters:     poBufferQ       The buffer queue.
                uiTimeoutMsec   If the queue is empty, wait the until a new buffer is pushed until this
                                timeout expired. No waiting when set to 0.
                poData          Poped data.

Return:         MTRUE on succeed.
                MFALSE if the queue is empty.

\************************************************************************************************************/
MBOOL BufQ_Pop(
        BufferQueue*            poBufferQ,
        MUINT                   uiTimeoutMsec,
        PopData*                poData)
{
    MBOOL bDone = MFALSE;

    MsgLog(8, "{...");

    LMutex_Lock(poBufferQ->hMutex);

    if (BufQ_IsEmpty(poBufferQ) && (uiTimeoutMsec > 0))
    {
        LCondition_Wait(poBufferQ->hNotEmptyCond, poBufferQ->hMutex, uiTimeoutMsec);
    }

    if (!BufQ_IsEmpty(poBufferQ))
    {
        poData->oElement = poBufferQ->aoElement[poBufferQ->uiPopIdx];

        MsgLog(6, "Poped: Buffer[%u], Tag= %ld, DT= %p", poData->oElement.poBufferInfo->uiId,
                  poData->oElement.uiTag, poData->oElement.hDeviceThread);

        poBufferQ->uiPopIdx++;

        if (poBufferQ->uiPopIdx >= poBufferQ->uiElementCount)
        {
            poBufferQ->uiPopIdx   = 0;
            poBufferQ->uiPushIdx -= poBufferQ->uiElementCount;
        }

        poData->uiQueueSize         = poBufferQ->uiPushIdx - poBufferQ->uiPopIdx;
        poData->uiQueueLatencyUsec  = poBufferQ->uiLastTimestampUsec
                                      - poData->oElement.poBufferInfo->uiTimestampUsec;

        bDone = MTRUE;
    }

    LMutex_Unlock(poBufferQ->hMutex);

    MsgLog(8, "...} (ret= %s)", bDone ? "true" : "false");

    return bDone;
}

/************************************************************************************************************\

Function:       BufQ_Peek

Description:    Get a buffer from the queue head without removing it.

Parameters:     poBufferQ       The buffer queue.
                poData          Peeked data.

Return:         MTRUE on succeed.
                MFALSE if the queue is empty.

\************************************************************************************************************/
MBOOL BufQ_Peek(
        BufferQueue*    poBufferQ,
        PopData*        poData)
{
    MBOOL bDone = MFALSE;

    MsgLog(8, "{...");

    LMutex_Lock(poBufferQ->hMutex);

    if (!BufQ_IsEmpty(poBufferQ))
    {
        poData->oElement            = poBufferQ->aoElement[poBufferQ->uiPopIdx];
        poData->uiQueueSize         = poBufferQ->uiPushIdx - poBufferQ->uiPopIdx;
        poData->uiQueueLatencyUsec  = poBufferQ->uiLastTimestampUsec
                                      - poData->oElement.poBufferInfo->uiTimestampUsec;
        bDone = MTRUE;
    }

    LMutex_Unlock(poBufferQ->hMutex);

    MsgLog(8, "...} (ret= %s)", bDone ? "true" : "false");

    return bDone;
}
